import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Camera, Star, BookOpen, User, Calendar, Award, Zap, TrendingUp, Users } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyCourses.css'

const MyCourses = () => {
  const [courses, setCourses] = useState([])

  useEffect(() => {
    loadCourses()
    const interval = setInterval(loadCourses, 500)
    return () => clearInterval(interval)
  }, [])

  const loadCourses = () => {
    const saved = localStorage.getItem('photoCourses')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setCourses(parsed)
      } catch (e) {
        setCourses([])
      }
    } else {
      const defaultCourses = [
        {
          id: 1,
          module: 'Основы композиции',
          instructor: 'Анна Петрова',
          completionDate: '2025-03-25',
          rating: 5
        },
        {
          id: 2,
          module: 'Работа со светом',
          instructor: 'Михаил Соколов',
          completionDate: '2025-03-20',
          rating: 5
        },
        {
          id: 3,
          module: 'Портретная съемка',
          instructor: 'Елена Волкова',
          completionDate: '2025-03-15',
          rating: 4
        }
      ]
      setCourses(defaultCourses)
      localStorage.setItem('photoCourses', JSON.stringify(defaultCourses))
    }
  }

  const totalCourses = courses.length
  const uniqueInstructors = new Set(courses.map(c => c.instructor)).size
  const averageRating = courses.length > 0 
    ? (courses.reduce((sum, c) => sum + c.rating, 0) / courses.length).toFixed(1)
    : 0

  return (
    <div className="my-courses">
      <div className="container">
        <motion.section 
          className="hero-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="hero-icon-wrapper"
            initial={{ scale: 0, rotate: -180 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
          >
            <Camera size={100} strokeWidth={2.5} className="hero-icon" />
          </motion.div>
          <h1 className="hero-title">Мои курсы</h1>
          <p className="hero-description">
            Ваш прогресс в обучении цифровой фотографии и достижения
          </p>
        </motion.section>

        <motion.div 
          className="stats-grid"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <motion.div 
            className="stat-card"
            whileHover={{ y: -4, transition: { duration: 0.2 } }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(139, 92, 246, 0.2)' }}>
              <BookOpen size={48} strokeWidth={2.5} color="#8B5CF6" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{totalCourses}</div>
              <div className="stat-label">Курсов</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ y: -4, transition: { duration: 0.2 } }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(236, 72, 153, 0.2)' }}>
              <Star size={48} strokeWidth={2.5} color="#EC4899" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{averageRating}</div>
              <div className="stat-label">Средняя оценка</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ y: -4, transition: { duration: 0.2 } }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(245, 158, 11, 0.2)' }}>
              <User size={48} strokeWidth={2.5} color="#F59E0B" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{uniqueInstructors}</div>
              <div className="stat-label">Преподавателей</div>
            </div>
          </motion.div>
        </motion.div>

        <ImageSlider images={[
          '/images/course-1.jpg',
          '/images/course-2.jpg',
          '/images/course-3.jpg',
          '/images/course-4.jpg'
        ]} />

        <motion.section 
          className="courses-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title">Ваши курсы</h2>
          {courses.length > 0 ? (
            <div className="courses-grid">
              {courses.map((course, index) => (
                <motion.div 
                  key={course.id} 
                  className="course-card"
                  initial={{ opacity: 0, scale: 0.9, rotate: -5 }}
                  animate={{ opacity: 1, scale: 1, rotate: 0 }}
                  transition={{ delay: index * 0.15, duration: 0.5 }}
                  whileHover={{ 
                    y: -4,
                    transition: { duration: 0.2 }
                  }}
                  style={{ 
                    borderLeft: '6px solid var(--primary)',
                    background: 'linear-gradient(135deg, rgba(15, 15, 35, 0.99) 0%, rgba(20, 20, 40, 0.98) 100%)'
                  }}
                >
                  <div className="course-header">
                    <div className="course-module-badge" style={{ background: 'rgba(139, 92, 246, 0.25)', color: '#8B5CF6' }}>
                      {course.module}
                    </div>
                    <div className="course-rating">
                      {Array.from({ length: 5 }, (_, i) => (
                        <Star
                          key={i}
                          size={20}
                          fill={i < course.rating ? '#F59E0B' : 'none'}
                          stroke={i < course.rating ? '#F59E0B' : '#666666'}
                          strokeWidth={2}
                        />
                      ))}
                    </div>
                  </div>
                  <div className="course-content">
                    <div className="course-info-item">
                      <User size={24} strokeWidth={2.5} color="#8B5CF6" />
                      <span className="course-info-text">{course.instructor}</span>
                    </div>
                    <div className="course-info-item">
                      <Calendar size={24} strokeWidth={2.5} color="#EC4899" />
                      <span className="course-info-text">
                        Завершен: {new Date(course.completionDate).toLocaleDateString('ru-RU', { 
                          day: 'numeric', 
                          month: 'long', 
                          year: 'numeric' 
                        })}
                      </span>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <div className="empty-icon-circle">
                <Camera size={80} strokeWidth={1.5} />
              </div>
              <h3 className="empty-title">Нет курсов</h3>
              <p className="empty-text">Запишитесь на курс прямо сейчас</p>
            </div>
          )}
        </motion.section>
      </div>
    </div>
  )
}

export default MyCourses

